<?php

namespace App\Http\Controllers\V1\QueryTraits;

use App\Models\ClientCollection;
use App\Models\Hims\HydrologySwlData;
use DateTime;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Facades\DB;

trait SwManualApiQuery
{
    public function getSwManualDataQuery($data, $clientCollectionId, $paginate = false)
    {
        $Datas = HydrologySwlData::query()
            ->join(DB::raw('HYDROLOGY.SW_STATION_TBL@HYDRO10 sw_station_tbl'), function (JoinClause $join) {
                $join->on('swl_data.station_id', '=', 'sw_station_tbl.station_id');
            })
            ->with('Station')
            ->whereHas('Station', function ($query) use ($data, $clientCollectionId) {
                if (ClientCollection::whereNull('is_all_station')->find($clientCollectionId)) {
                    $query->whereHas('ClientCollectionStation', function ($query) use ($clientCollectionId) {
                        $query->where('api_client_collection_stations.client_collection_id', $clientCollectionId); // 21 is for surface water
                    });
                }

                if (isset($data['identifier'])) {
                    $query->where('station_id', $data['identifier']);
                }
            })
        ;

        $Datas->select(
            'swl_data.station_id',
            DB::raw('TO_CHAR(swl_data.swl_date, \'DD/MM/YYYY\') as formatted_swl_date'),
            DB::raw('TO_CHAR(sw_station_tbl.sob_date, \'DD/MM/YYYY\') as formatted_sob_date'),
            'sw_station_tbl.rt',
            DB::raw('ROUND(AVG(swl_data.wl),2) as daily_avg_wl_round'),
            DB::raw('ROUND(MAX(swl_data.wl),2) as daily_max_wl_round'),
            DB::raw('ROUND(MIN(swl_data.wl),2) as daily_min_wl_round')
        );
        $Datas->groupBy(
            DB::raw('TO_CHAR(swl_data.swl_date, \'DD/MM/YYYY\')'),
            DB::raw('TO_CHAR(sw_station_tbl.sob_date, \'DD/MM/YYYY\')'),
            'sw_station_tbl.rt', // Include rt in the group by if needed
            'swl_data.station_id'
        );

        if (isset($data['identifier'])) {
            $Datas->where('swl_data.station_id', $data['identifier']);
        }

        if (isset($data['date']) && $data['date']) {
            $Datas->whereDate('swl_data.swl_date', $data['date']);
        } else {
            if (isset($data['from_date']) && $data['from_date']) {
                $Datas->whereDate('swl_data.swl_date', '>=', $data['from_date']);
            }

            if (isset($data['to_date']) && $data['to_date']) {
                $Datas->whereDate('swl_data.swl_date', '<=', $data['to_date']);
            }
        }

        if (isset($data['per_page']) && !$paginate) {
            $Datas->limit($data['per_page']);
        }

        if (!isset($data['per_page'])) {
            $data['per_page'] = 10;
        }

        // if ($paginate) {
        //     return $Datas->paginate($data['per_page']);
        // }

        $returnData = $Datas->get()->sortBy('formatted_swl_date')->map(function ($item) {
            $swl_date = DateTime::createFromFormat('d/m/yy', $item->formatted_swl_date);
            $sob_date = DateTime::createFromFormat('d/m/yy', $item->formatted_sob_date);
            $item->swl_date = $swl_date;
            $item->sob_date = $sob_date;

            $avg_value = 0;

            if ($swl_date >= $sob_date) {
                $avg_value = $item->daily_avg_wl_round;
                $min_value = $item->daily_min_wl_round;
                $max_value = $item->daily_max_wl_round;
            } elseif ($item->formatted_sob_date === null && $item->rt === null) {
                $avg_value = $item->daily_avg_wl_round;
                $min_value = $item->daily_min_wl_round;
                $max_value = $item->daily_max_wl_round;
            } else {
                $avg_value = $item->daily_avg_wl_round + $item->rt;
                $min_value = $item->daily_min_wl_round + $item->rt;
                $max_value = $item->daily_max_wl_round + $item->rt;
            }

            $item->value = round($avg_value, 2);
            $item->value_min = round($min_value, 2);
            $item->value_max = round($max_value, 2);

            return $item;
        });

        return $returnData;
    }
}
