<?php
namespace App\Http\Controllers\V1;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Traits\CheckUser;
use App\Models\Hidm\FfwcDataTbl;
use App\Models\Hidm\HydroDataLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class StationDataFetchController extends Controller
{
    use CheckUser;
    public function hydroDatafetch(Request $request)
    {
        if ($this->checkUser("hydrosms") == 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized Access',
            ], 401);
        }

        $validation = [
            'keyword' => 'required', 'string',
            'date' => 'required',
        ];

        $validator = Validator::make($request->all(), $validation);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => "Validation Error",
                'errors' => $validator->errors(),
            ], 422);
        } else {
            $date = now()->parse($request->date);
            $data = HydroDataLog::where('well_id', $request->keyword)->whereDate('log_date', $date)->first();
            return response()->json([
                'status' => 'success',
                'value' => isset($data->water_lvl) ? $data->water_lvl : '',
            ], 200);
        }
    }

    public function ffwcDatafetch(Request $request)
    {

        if ($this->checkUser("ffwcsms") == 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized Access',
            ], 401);
        }

        $validation = [
            'keyword' => 'required', 'string',
            'date' => 'required',
        ];

        $validator = Validator::make($request->all(), $validation);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => "Validation Error",
                'errors' => $validator->errors(),
            ], 422);
        }

        $date = now()->parse($request->date);

        $dataHr = FfwcDataTbl::where(DB::raw('UPPER(station_code)'), strtoupper($request->keyword))
            ->whereDate('m_date', $date)
            ->where('d_type', 'W')
            ->get()
            ->sortBy('m_date')
            ->mapWithKeys(function ($items) {
                $hour = date('H', strtotime($items->m_date)); // Extract hour from m_date
                $key = 'hr' . (int) $hour;                    // Create dynamic key (e.g., hr6, hr9)
                return [$key => $items->data_value];
            })
            ->toArray();

        foreach ([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23] as $value) {
            if (! isset($dataHr['hr' . $value])) {
                $dataHr['hr' . $value] = null;
            }
        }

        $dataRainfall = FfwcDataTbl::where(DB::raw('UPPER(station_code)'), strtoupper($request->keyword))
            ->whereDate('m_date', $date)
            ->where('d_type', 'R')
            ->get()
            ->sortBy('m_date')
            ->mapWithKeys(function ($items) {
                return [
                    'rainfall' => $items->data_value,
                ];
            })
            ->toArray();

        if (! isset($dataRainfall['rainfall'])) {
            $dataRainfall['rainfall'] = null;
        }

        $data = array_merge(['status' => 'success'], $dataHr, $dataRainfall);

        return response()->json($data, 200);

    }

    public function ffwcDatafetchBack(Request $request)
    {

        if ($this->checkUser("ffwcsms") == 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized Access',
            ], 401);
        }

        $validation = [
            'keyword' => 'required', 'string',
            'from_date' => 'required',
            'to_date' => 'required',
        ];

        $validator = Validator::make($request->all(), $validation);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => "Validation Error",
                'errors' => $validator->errors(),
            ], 422);
        }

        $from_date = now()->parse($request->from_date);
        $to_date = now()->parse($request->to_date);

        if ($from_date->diffInDays($to_date) >= 8) {
            return response()->json([
                'status' => 'error',
                'message' => "Date need to be in between 7 days",
            ], 422);
        }

        $dataHr = FfwcDataTbl::where(DB::raw('UPPER(station_code)'), strtoupper($request->keyword))
            ->whereDate('m_date', '>=', $from_date)
            ->whereDate('m_date', '<=', $to_date)
            ->where('d_type', 'W')
            ->get()
            ->sortBy('m_date')
            ->groupBy(function ($item) {
                return $item->m_date->format('d-m-Y'); // Group by date
            })
            ->map(function ($items) {
                return $items->mapWithKeys(function ($item) {
                    $hour = date('H', strtotime($item->m_date)); // Extract hour from m_date
                    $key = 'hr' . (int) $hour;                   // Create dynamic key (e.g., hr6, hr9)

                    return [$key => $item->data_value];
                });
            })
            ->toArray();

        $dataRainfall = FfwcDataTbl::where(DB::raw('UPPER(station_code)'), strtoupper($request->keyword))
            ->whereDate('m_date', '>=', $from_date)
            ->whereDate('m_date', '<=', $to_date)
            ->where('d_type', 'R')
            ->get()
            ->sortBy('m_date')
            ->groupBy(function ($item) {
                return $item->m_date->format('d-m-Y'); // Group by date
            })
            ->map(function ($items) {
                return $items->mapWithKeys(function ($item) {
                    return [
                        'rainfall' => $item->data_value,
                    ];
                });
            })
            ->toArray();

        $data = array_merge(['status' => 'success'], $dataHr, $dataRainfall);

        return response()->json($data, 200);

    }

    public function hydroGraphData(Request $request)
    {
        if ($this->checkUser("hydrosms") == 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized Access',
            ], 401);
        }
        $from_date = now()->parse($request->from_date);
        $to_date = now()->parse($request->to_date);
        if ($from_date->diffInDays($to_date) >= 30) {
            return response()->json([
                'status' => 'error',
                'message' => "Date need to be in between 30 days",
            ], 422);
        }

        $validation = [
            'keyword' => 'required', 'string',
            'from_date' => 'required',
            'to_date' => 'required',
        ];

        $validator = Validator::make($request->all(), $validation);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => "Validation Error",
                'errors' => $validator->errors(),
            ], 422);
        } else {

            $data = HydroDataLog::select('log_date', 'water_lvl')->where('well_id', $request->keyword)
                ->whereDate('log_date', '>=', $from_date)
                ->whereDate('log_date', '<=', $to_date)
                ->orderBy('log_date', 'asc')->get()
                ->map(function ($item) {
                    $item->log_date = Carbon::parse($item->log_date)->format('d-m-Y');
                    return $item;
                });

            return response()->json([
                'status' => 'success',
                'data' => $data,
            ], 200);
        }
    }
    public function ffwcGraphData(Request $request)
    {
        if ($this->checkUser("ffwcsms") == 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized Access',
            ], 401);
        }
        $from_date = now()->parse($request->from_date);
        $to_date = now()->parse($request->to_date);
        if ($from_date->diffInDays($to_date) >= 8) {
            return response()->json([
                'status' => 'error',
                'message' => "Date need to be in between 7 days",
            ], 422);
        }

        $validation = [
            'keyword' => ['required', 'string'],
            'from_date' => ['required', 'date'],
            'to_date' => ['required', 'date'],
            'type' => ['required', 'string'],
        ];

        $validator = Validator::make($request->all(), $validation);
        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => "Validation Error",
                'errors' => $validator->errors(),
            ], 422);
        } else {

            if ($request->type == 'RF') {
                $data = FfwcDataTbl::where(DB::raw('UPPER(station_code)'), strtoupper($request->keyword))
                    ->whereDate('m_date', '>=', $from_date)
                    ->whereDate('m_date', '<=', $to_date)
                    ->where('d_type', 'R')
                    ->orderBy('m_date', 'asc')
                    ->get()
                    ->map(function ($item) {
                        return [
                            'log_date' => Carbon::parse($item->m_date)->format('d-m-Y'),
                            'rainfall' => $item->data_value,
                        ];
                    })->toArray();
            } elseif ($request->type == 'WL') {
                $data = FfwcDataTbl::where(DB::raw('UPPER(station_code)'), strtoupper($request->keyword))
                    ->whereDate('m_date', '>=', $from_date)
                    ->whereDate('m_date', '<=', $to_date)
                    ->where('d_type', 'W')
                    ->orderBy('m_date', 'asc')
                    ->get()
                    ->groupBy(function ($item) {
                        return date('Y-m-d', strtotime($item->m_date)); // Group by date
                    })
                    ->map(function ($items) {
                        $formattedData = [];

                        foreach ($items as $item) {

                            $hour = (int) date('H', strtotime($item->m_date));                         // Extract hour
                            $key = "hr{$hour}";                                                        // Create key (e.g., hr6, hr9)
                            $formattedData['log_date'] = now()->parse($item->m_date)->format('d-m-Y'); // Store data
                            $formattedData[$key] = $item->data_value;                                  // Store data
                        }

                        foreach ([6, 9, 12, 15, 18] as $value) {
                            if (! isset($formattedData['hr' . $value])) {
                                $formattedData['hr' . $value] = null;
                            }
                        }

                        uksort($formattedData, function ($a, $b) {
                            $order = ['log_date', 'hr6', 'hr9', 'hr12', 'hr15', 'hr18'];
                            return array_search($a, $order) - array_search($b, $order);
                        });

                        return $formattedData;
                    })
                    ->toArray();
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Type Not Match',
                ], 401);
            }

            return response()->json([
                'status' => 'success',
                "$request->type Data" => $data,
            ], 200);
        }
    }
}
