<?php

namespace App\Http\Controllers\V1;

use App\Http\Controllers\V1\ApiController;
use App\Http\Controllers\V1\QueryTraits\SwApiQuery;
use App\Http\Resources\RainfallAutoDataResource;
use App\Http\Resources\RainfallStationResource;
use App\Http\Resources\SwByTypeManualStation;
use App\Http\Resources\SwManualStation;
use App\Http\Resources\WaterlevelStationResource;
use App\Http\Resources\WeatherAutoDataResource;
use App\Models\ClientCollection;
use App\Models\ClientCollectionParameter;
use App\Models\Hims\FfwcStation;
use App\Models\Hims\RainfallAutoData;
use App\Models\Hims\SwStationManual;
use App\Models\Hims\WeatherAutoData;
use App\Models\Hims\WlAutuData;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StationController extends ApiController
{
    use SwApiQuery;

    public function wlStationJson(Request $request)
    {
        $ClientCollection = ClientCollection::whereHas('ClientToken', function ($query) {
            return $query->where('api_client_tokens.token', request()->bearerToken());
        })->active()->where('collection_id', 5)->first();

        if (!$ClientCollection) {
            return response()->json([
                'message' => 'You are not authorized to access this API',
                'status' => 'Unauthorized',
            ], 403);
        }

        $log = $this->apiLogRequest($ClientCollection);

        $RequestData = [];

        if ($request->has('from_date')) {
            $RequestData['from_date'] = $request->from_date;
        } else {
            $RequestData['from_date'] = now()->format('Y-m-d');
        }

        if ($request->has('to_date')) {
            $RequestData['to_date'] = $request->to_date;
        } else {
            $RequestData['to_date'] = now()->format('Y-m-d');
        }

        if ($request->has('station_id')) {
            $RequestData['identifier'] = $request->station_id;
        } elseif ($request->has('station')) {
            $RequestData['identifier'] = $request->station;
        } elseif ($request->has('identifier')) {
            $RequestData['identifier'] = $request->identifier;
        }

        if ($request->has('per_page')) {
            $RequestData['per_page'] = $request->per_page;
        }

        $ClientCollectionParameter = ClientCollectionParameter::with('CollectionParameter')->where('collection_id', $ClientCollection->collection_id)->where('client_id', $ClientCollection->client_id)->get();

        foreach ($ClientCollectionParameter as $key => $value) {
            if (isset($RequestData['per_page']) && $value->CollectionParameter->name == 'max_per_page') {
                if ($RequestData['per_page'] > $value->value) {
                    return response()->json([
                        'message' => 'Your per page limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }

            if (isset($RequestData['from_date']) && $value->CollectionParameter->name == 'start_date') {
                if (now()->parse($RequestData['from_date'])->format('Y-m-d') < now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your from date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }
        }

        $Data = FfwcStation::query()
            ->where('ffwc_common', 'FFWC')
            ->whereIn('ffwc_station_type', [0, 1]);
        // dd(count($Data->get()));

        $perPage = 10;

        if (isset($RequestData['per_page']) && $RequestData['per_page'] > 0) {
            $perPage = $RequestData['per_page'];
        }

        if (isset($RequestData['identifier'])) {
            $Data->where('station_id', $RequestData['identifier']);

            $ReturnData = $Data->first();

            if ($ReturnData) {
                $ReturnData = new WaterlevelStationResource($ReturnData);

                if ($ReturnData) {
                    $this->apiLogResponse($log);
                }

                return $ReturnData;
            } else {
                return ['message' => 'No data found', 'status' => 'success'];
            }
        }

        $ReturnData = WaterlevelStationResource::collection($Data->paginate($perPage))->additional($this->commonJsonReturn('You surface water data json format is successfully'));

        if ($ReturnData) {
            $this->apiLogResponse($log);
        }

        return $ReturnData;
    }

    public function rfStationJson(Request $request)
    {
        $ClientCollection = ClientCollection::whereHas('ClientToken', function ($query) {
            return $query->where('api_client_tokens.token', request()->bearerToken());
        })->active()->where('collection_id', 6)->first();

        if (!$ClientCollection) {
            return response()->json([
                'message' => 'You are not authorized to access this API',
                'status' => 'Unauthorized',
            ], 403);
        }

        $RequestData = [];

        if ($request->has('from_date')) {
            $RequestData['from_date'] = $request->from_date;
        } else {
            $RequestData['from_date'] = now()->format('Y-m-d');
        }

        if ($request->has('to_date')) {
            $RequestData['to_date'] = $request->to_date;
        } else {
            $RequestData['to_date'] = now()->format('Y-m-d');
        }

        if ($request->has('station_id')) {
            $RequestData['identifier'] = $request->station_id;
        } elseif ($request->has('station')) {
            $RequestData['identifier'] = $request->station;
        } elseif ($request->has('identifier')) {
            $RequestData['identifier'] = $request->identifier;
        }

        if ($request->has('per_page')) {
            $RequestData['per_page'] = $request->per_page;
        }

        $ClientCollectionParameter = ClientCollectionParameter::with('CollectionParameter')->where('collection_id', $ClientCollection->collection_id)->where('client_id', $ClientCollection->client_id)->get();

        foreach ($ClientCollectionParameter as $key => $value) {
            if (isset($RequestData['per_page']) && $value->CollectionParameter->name == 'max_per_page') {
                if ($RequestData['per_page'] > $value->value) {
                    return response()->json([
                        'message' => 'Your per page limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }

            if (isset($RequestData['from_date']) && $value->CollectionParameter->name == 'start_date') {
                if (now()->parse($RequestData['from_date'])->format('Y-m-d') < now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your from date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }
        }

        $Data = FfwcStation::query()
            ->where('ffwc_common', 'FFWC')
            ->whereIn('ffwc_station_type', [0, 2]);

        $perPage = 10;

        if (isset($RequestData['per_page']) && $RequestData['per_page'] > 0) {
            $perPage = $RequestData['per_page'];
        }

        if (isset($RequestData['identifier'])) {
            $Data->where('station_code', $RequestData['identifier']);

            $ReturnData = $Data->first();

            if ($ReturnData) {
                return new RainfallStationResource($ReturnData);
            } else {
                return ['message' => 'No data found', 'status' => 'success'];
            }
        }

        return RainfallStationResource::collection($Data->paginate($perPage))->additional($this->commonJsonReturn('You surface water data json format is successfully'));
    }

    public function waterLevelData(Request $request)
    {
        $from_date = null;
        $to_date = null;
        if ($request->has('from_date')) {
            $from_date = now()->parse($request->from_date);
        } else {
            $from_date = now();
        }

        if ($request->has('to_date')) {
            $to_date = now()->parse($request->to_date);
        } else {
            $to_date = now();
        }

        if ($request->has('station_id')) {
            $data = WlAutuData::whereDate('time', '>=', $from_date)->whereDate('time', '<=', $to_date)
                ->where('identifier', $request->station_id)->orderBy('time')
                ->get()
                ->map(function ($item) {
                    return [
                        'Station ID' => $item->identifier,
                        'Date Time' => $item->main_time,
                        "Water Level" => $item->value,
                        "Measurement Unit" => 'm',

                    ];
                });
        } else {
            $data = WlAutuData::whereDate('time', '>=', $from_date)->whereDate('time', '<=', $to_date)
                ->where('identifier', 'like', 'SW%')->orderBy('time')
                ->get()
                ->map(function ($item) {
                    return [
                        'Station ID' => $item->identifier,
                        'Date Time' => $item->main_time,
                        "Water Level" => $item->value,
                        "Measurement Unit" => 'm',

                    ];
                });
        }

        return response()->json([
            'status' => 'success',
            'surface_water_list' => $data,
        ], 200);
    }

    public function rainFallData(Request $request)
    {

        $ClientCollection = ClientCollection::whereHas('ClientToken', function ($query) {
            return $query->where('api_client_tokens.token', request()->bearerToken());
        })->active()->where('collection_id', 9)->first();

        if (!$ClientCollection) {
            return response()->json([
                'message' => 'You are not authorized to access this API',
                'status' => 'Unauthorized',
            ], 403);
        }

        $this->apiLogRequest($ClientCollection);

        $RequestData = [];

        if ($request->has('date')) {
            $RequestData['date'] = $request->date;
        } else {
            if ($request->has('from_date')) {
                $RequestData['from_date'] = $request->from_date;
            }

            if ($request->has('to_date')) {
                $RequestData['to_date'] = $request->to_date;
            }
        }

        if ($request->has('station_id')) {
            $RequestData['identifier'] = $request->station_id;
        } elseif ($request->has('station')) {
            $RequestData['identifier'] = $request->station;
        } elseif ($request->has('identifier')) {
            $RequestData['identifier'] = $request->identifier;
        }

        if ($request->has('per_page')) {
            $RequestData['per_page'] = $request->per_page;
        }

        $ClientCollectionParameter = ClientCollectionParameter::with('CollectionParameter')->where('collection_id', $ClientCollection->collection_id)->where('client_id', $ClientCollection->client_id)->get();

        foreach ($ClientCollectionParameter as $key => $value) {

            if (isset($RequestData['per_page']) && $value->CollectionParameter->name == 'max_per_page') {
                if ($RequestData['per_page'] > $value->value) {
                    return response()->json([
                        'message' => 'Your per page limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }

            }

            if (isset($RequestData['from_date']) && $value->CollectionParameter->name == 'start_date' && $value->value) {
                if (now()->parse($RequestData['from_date'])->format('Y-m-d') < now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your from date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }

            if (isset($RequestData['to_date']) && $value->CollectionParameter->name == 'end_date' && $value->value) {
                if (now()->parse($RequestData['to_date'])->format('Y-m-d') >= now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your to date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }
        }

        $Data = RainfallAutoData::query();

        if (isset($RequestData['date']) && $RequestData['date']) {
            $Data->whereDate('time', '>=', $RequestData['date']);
        } else {
            if (isset($RequestData['from_date']) && $RequestData['from_date']) {
                $Data->whereDate('time', '>=', $RequestData['from_date']);
            }

            if (isset($RequestData['to_date']) && $RequestData['to_date']) {
                $Data->whereDate('time', '<=', $RequestData['to_date']);
            }
        }

        $perPage = 10;

        if (isset($RequestData['per_page']) && $RequestData['per_page'] > 0) {
            $perPage = $RequestData['per_page'];
        }

        if (isset($RequestData['identifier'])) {
            $Data->where(DB::raw("LOWER(identifier)"), strtolower($RequestData['identifier']));
        } else {
            $Data->where(DB::raw("LOWER(identifier)"), 'like', 'cl%');
        }

        return RainfallAutoDataResource::collection($Data->paginate($perPage))->additional($this->commonJsonReturn('You rainfall auto data json format is successfully'));

    }

    public function weatherAutoData(Request $request)
    {

        $ClientCollection = ClientCollection::whereHas('ClientToken', function ($query) {
            return $query->where('api_client_tokens.token', request()->bearerToken());
        })->active()->where('collection_id', 11)->first();

        if (!$ClientCollection) {
            return response()->json([
                'message' => 'You are not authorized to access this API',
                'status' => 'Unauthorized',
            ], 403);
        }

        $this->apiLogRequest($ClientCollection);

        $RequestData = [];

        if ($request->has('date')) {
            $RequestData['date'] = $request->date;
        } else {
            if ($request->has('from_date')) {
                $RequestData['from_date'] = $request->from_date;
            }

            if ($request->has('to_date')) {
                $RequestData['to_date'] = $request->to_date;
            }
        }

        if ($request->has('station_id')) {
            $RequestData['identifier'] = $request->station_id;
        } elseif ($request->has('station')) {
            $RequestData['identifier'] = $request->station;
        } elseif ($request->has('identifier')) {
            $RequestData['identifier'] = $request->identifier;
        }

        if ($request->has('per_page')) {
            $RequestData['per_page'] = $request->per_page;
        }

        $ClientCollectionParameter = ClientCollectionParameter::with('CollectionParameter')->where('collection_id', $ClientCollection->collection_id)->where('client_id', $ClientCollection->client_id)->get();

        foreach ($ClientCollectionParameter as $key => $value) {

            if (isset($RequestData['per_page']) && $value->CollectionParameter->name == 'max_per_page') {
                if ($RequestData['per_page'] > $value->value) {
                    return response()->json([
                        'message' => 'Your per page limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }

            }

            if (isset($RequestData['from_date']) && $value->CollectionParameter->name == 'start_date' && $value->value) {
                if (now()->parse($RequestData['from_date'])->format('Y-m-d') < now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your from date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }

            if (isset($RequestData['to_date']) && $value->CollectionParameter->name == 'end_date' && $value->value) {
                if (now()->parse($RequestData['to_date'])->format('Y-m-d') >= now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your to date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }
        }

        $Data = WeatherAutoData::query();

        if (isset($RequestData['date']) && $RequestData['date']) {
            $Data->whereDate('bd_time', '>=', $RequestData['date']);
        } else {
            if (isset($RequestData['from_date']) && $RequestData['from_date']) {
                $Data->whereDate('bd_time', '>=', $RequestData['from_date']);
            }

            if (isset($RequestData['to_date']) && $RequestData['to_date']) {
                $Data->whereDate('bd_time', '<=', $RequestData['to_date']);
            }
        }

        $perPage = 10;

        if (isset($RequestData['per_page']) && $RequestData['per_page'] > 0) {
            $perPage = $RequestData['per_page'];
        }

        if (isset($RequestData['identifier'])) {
            $Data->where(DB::raw("LOWER(station_id)"), strtolower($RequestData['identifier']));
        } else {
            $Data->where(DB::raw("LOWER(station_id)"), 'like', 'cl%');
        }

        return WeatherAutoDataResource::collection($Data->paginate($perPage))->additional($this->commonJsonReturn('You rainfall auto data json format is successfully'));

    }

    public function stationListARF(Request $request)
    {
        $data = DB::select("select station_id,station_name,latitude,longitude,upzl_name,district_name
        from sw_station_auto_manual where data_read_type in ('AUTO','BOTH') and (',' || STATION_TYPE || ',' LIKE '%,' || '2' ||',%' OR ',' || STATION_TYPE || ',' LIKE '%,' || '10' ||',%')");

        return response()->json([
            'data' => $data,
            'version' => '1.0',
            'message' => 'You rainfall auto station list data json format is successfully',
            'status' => 'suceess',
        ], 200);
    }

    public function swManualStationJson(Request $request)
    {
        $ClientCollection = ClientCollection::whereHas('ClientToken', function ($query) {
            return $query->where('api_client_tokens.token', request()->bearerToken());
        })->active()->where('collection_id', 13)->first();

        if (!$ClientCollection) {
            return response()->json([
                'message' => 'You are not authorized to access this API',
                'status' => 'Unauthorized',
            ], 403);
        }

        $RequestData = [];

        if ($request->has('from_date')) {
            $RequestData['from_date'] = $request->from_date;
        } else {
            $RequestData['from_date'] = now()->format('Y-m-d');
        }

        if ($request->has('to_date')) {
            $RequestData['to_date'] = $request->to_date;
        } else {
            $RequestData['to_date'] = now()->format('Y-m-d');
        }

        if ($request->has('station_id')) {
            $RequestData['identifier'] = $request->station_id;
        } elseif ($request->has('station')) {
            $RequestData['identifier'] = $request->station;
        } elseif ($request->has('identifier')) {
            $RequestData['identifier'] = $request->identifier;
        }

        if ($request->has('per_page')) {
            $RequestData['per_page'] = $request->per_page;
        }

        $ClientCollectionParameter = ClientCollectionParameter::with('CollectionParameter')->where('collection_id', $ClientCollection->collection_id)->where('client_id', $ClientCollection->client_id)->get();

        foreach ($ClientCollectionParameter as $key => $value) {
            if (isset($RequestData['per_page']) && $value->CollectionParameter->name == 'max_per_page') {
                if ($RequestData['per_page'] > $value->value) {
                    return response()->json([
                        'message' => 'Your per page limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }

            if (isset($RequestData['from_date']) && $value->CollectionParameter->name == 'start_date') {
                if (now()->parse($RequestData['from_date'])->format('Y-m-d') < now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your from date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }
        }

        $Data = FfwcStation::query()
            ->whereIn('data_read_type', ['MANUAL', 'BOTH'])
        // ->whereRaw("INSTR(',' || \"station_type\" || ',', ',1,') > 0")
        // ->whereRaw("INSTR(',' || \"station_type\" || ',', ',1,') > 0")
            ->whereRaw("INSTR(',' || station_type || ',', ',1,') > 0")
            ->where('operation_status', 'Open');

        $perPage = 10;

        if (isset($RequestData['per_page']) && $RequestData['per_page'] > 0) {
            $perPage = $RequestData['per_page'];
        }

        if (isset($RequestData['identifier'])) {
            $Data->where('station_code', $RequestData['identifier']);

            $ReturnData = $Data->first();

            if ($ReturnData) {
                return new SwManualStation($ReturnData);
            } else {
                return ['message' => 'No data found', 'status' => 'success'];
            }
        }

        return SwManualStation::collection($Data->paginate($perPage))->additional($this->commonJsonReturn('You surface water data json format is successfully'));
    }

    public function swStationByTypeJson(Request $request)
    {
        $ClientCollection = ClientCollection::whereHas('ClientToken', function ($query) {
            return $query->where('api_client_tokens.token', request()->bearerToken());
        })->active()->where('collection_id', 15)->first();

        if (!$ClientCollection) {
            return response()->json([
                'message' => 'You are not authorized to access this API',
                'status' => 'Unauthorized',
            ], 403);
        }

        $RequestData = [];

        if ($request->has('from_date')) {
            $RequestData['from_date'] = $request->from_date;
        } else {
            $RequestData['from_date'] = now()->format('Y-m-d');
        }

        if ($request->has('to_date')) {
            $RequestData['to_date'] = $request->to_date;
        } else {
            $RequestData['to_date'] = now()->format('Y-m-d');
        }

        if ($request->has('station_id')) {
            $RequestData['identifier'] = $request->station_id;
        } elseif ($request->has('station')) {
            $RequestData['identifier'] = $request->station;
        } elseif ($request->has('identifier')) {
            $RequestData['identifier'] = $request->identifier;
        }

        if ($request->has('per_page')) {
            $RequestData['per_page'] = $request->per_page;
        }

        if ($request->has('data_type')) {
            $RequestData['data_type'] = $request->data_type;
        } else {
            return response()->json([
                'message' => 'Data type required',
                'status' => 'errors',
            ], 422);
        }

        $ClientCollectionParameter = ClientCollectionParameter::with('CollectionParameter')->where('collection_id', $ClientCollection->collection_id)->where('client_id', $ClientCollection->client_id)->get();

        foreach ($ClientCollectionParameter as $key => $value) {
            if (isset($RequestData['per_page']) && $value->CollectionParameter->name == 'max_per_page') {
                if ($RequestData['per_page'] > $value->value) {
                    return response()->json([
                        'message' => 'Your per page limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }

            if (isset($RequestData['from_date']) && $value->CollectionParameter->name == 'start_date') {
                if (now()->parse($RequestData['from_date'])->format('Y-m-d') < now()->parse($value->value)->format('Y-m-d')) {
                    return response()->json([
                        'message' => 'Your from date limit is ' . $value->value,
                        'status' => 'errors',
                    ], 422);
                }
            }
        }

        $stationTypeOr = null;
        if ($RequestData['data_type'] === 'rf') {
            $stationType = 2;
        } elseif ($RequestData['data_type'] === 'od') {
            $stationType = 4;
        } elseif ($RequestData['data_type'] === 'cs') {
            $stationType = 13;
        } elseif ($RequestData['data_type'] === 'sl') {
            $stationType = 6;
            $stationTypeOr = 14;
        } elseif ($RequestData['data_type'] === 'wl') {
            $stationType = 1;
        } elseif ($RequestData['data_type'] === 'td') {
            $stationType = 8;
        } elseif ($RequestData['data_type'] === 'sd') {
            $stationType = 5;
        }

        $Data = SwStationManual::query()
            ->where(function ($query) use ($stationType, $stationTypeOr) {
                $query->whereRaw("INSTR(',' || station_type || ',', ',{$stationType},') > 0");
                if ($stationTypeOr) {
                    $query->orWhereRaw("INSTR(',' || station_type || ',', ',{$stationTypeOr},') > 0");
                }
            })

        ;

        $perPage = 10;

        if (isset($RequestData['per_page']) && $RequestData['per_page'] > 0) {
            $perPage = $RequestData['per_page'];
        }

        if (isset($RequestData['identifier'])) {
            $Data->where('station_id', $RequestData['identifier']);

            $ReturnData = $Data->first();

            if ($ReturnData) {
                $returnData = new SwByTypeManualStation($ReturnData);
                return $returnData->additional($this->commonJsonReturn('You surface water station information json format is successfully'));
            } else {
                return ['message' => 'No data found', 'status' => 'success'];
            }
        }

        return SwByTypeManualStation::collection($Data->get()->sortBy('station_id'))->additional($this->commonJsonReturn('You surface water station information json format is successfully'));
    }
}
