<?php
// ------------------ Configuration Area ------------------ //
function is_image_ext($ext) {
    $img_exts = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'ico', 'bmp', 'svg'];
    return in_array(strtolower($ext), $img_exts);
}

function is_document_ext($ext) {
    $doc_exts = ['doc', 'docx', 'pdf', 'txt', 'xls', 'xlsx', 'ppt', 'pptx', 'odt', 'ods', 'odp'];
    return in_array(strtolower($ext), $doc_exts);
}

function is_font_ext($ext) {
    $font_exts = ['ttf', 'woff', 'woff2', 'eot', 'otf'];
    return in_array(strtolower($ext), $font_exts);
}

function is_css_ext($ext) {
    $css_exts = ['css', 'scss'];
    return in_array(strtolower($ext), $css_exts);
}

function is_ignored_file($filePath) {
    return stripos(basename($filePath), 'sess_') === 0;
}

function is_large_file($filePath, $limit_bytes = 1048576) {
    return filesize($filePath) > $limit_bytes;
}

function is_excluded_dir($filePath, $basePath) {
    $relative = str_replace('\\', '/', substr($filePath, strlen($basePath) + 1));
    $excluded_dirs = ['runtime/', 'cache/', '.git/', 'tmp/'];
    // Exclude directories containing 'debug' (case-insensitive)
    if (stripos($relative, 'debug') !== false) {
        return true;
    }
    foreach ($excluded_dirs as $ex) {
        if (strpos($relative, $ex) === 0 || strpos($relative, '/' . $ex) !== false) {
            return true;
        }
    }
    return false;
}
// ------------------ Main Program Entry ------------------ //

$_act = $_GET['act'] ?? '';
$_datetime = date("Ymd_His");
$_root = realpath($_SERVER['DOCUMENT_ROOT']); // root dir for display

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $_act === 'sync') {
    $_input_dir = trim($_POST['target_dir'] ?? '');
    $_zip_path = trim($_POST['zip_path'] ?? '');
    $_target = realpath($_input_dir);

    if (!$_target || !is_dir($_target)) {
        exit("Error: Invalid directory path");
    }

    // Determine zip filename: directory name + datetime + .zip
    $dir_name = basename($_target);
    $zip_name = $dir_name . "_{$_datetime}.zip";

    // Determine zip full path
    if ($_zip_path !== '') {
        // Make sure directory exists and is writable
        if (!is_dir($_zip_path) || !is_writable($_zip_path)) {
            exit("Error: Specified zip save path is invalid or not writable");
        }
        $_zipfile = rtrim($_zip_path, '/\\') . DIRECTORY_SEPARATOR . $zip_name;
    } else {
        $_zipfile = $zip_name; // current working dir
    }

    echo "<h3>Packing directory: <code>{$_target}</code></h3>";

    $zip = new ZipArchive();
    if ($zip->open($_zipfile, ZipArchive::CREATE | ZipArchive::OVERWRITE)) {
        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($_target, FilesystemIterator::SKIP_DOTS)
        );

        foreach ($it as $file) {
            $ext = pathinfo($file, PATHINFO_EXTENSION);
            $rel_path = substr($file, strlen($_target) + 1);

            if (is_image_ext($ext)) continue;
            if (is_ignored_file($file)) continue;
            if (is_excluded_dir($file, $_target)) continue;

            // Document files - show skip message
            if (is_document_ext($ext)) {
                echo "Skipped document file: $rel_path<br>";
                continue;
            }

            if (is_font_ext($ext)) continue;
            if (is_css_ext($ext)) continue;

            if (is_large_file($file)) {
                $size_mb = round(filesize($file) / 1048576, 2);
                echo "Skipped large file ({$size_mb}MB): $rel_path<br>";
                continue;
            }

            $zip->addFile($file, $rel_path);
            // Do not output success messages
        }

        $zip->close();
        echo "<br>Done: <a href='{$_zipfile}' download>Click to download archive</a>";
    } else {
        echo "Error: Failed to create zip file";
    }
} else {
    // Display input form with root dir info and zip save path input
    ?>
    <h2>Directory Packaging Tool</h2>
    <p>Current website root directory is: <code><?= htmlspecialchars($_root) ?></code></p>
    <form method="post" action="?act=sync">
        <label>Please enter the directory path to pack:</label><br>
        <input type="text" name="target_dir" value="<?= htmlspecialchars($_root) ?>" style="width:500px;"><br><br>

        <label>Optional: enter directory path to save the zip file (default current directory):</label><br>
        <input type="text" name="zip_path" placeholder="Leave empty to save in current directory" style="width:500px;"><br><br>

        <button type="submit">Start Packing</button>
    </form>
    <?php
}
?>
